import {
  AccessToken,
  HttpStatusCode,
  TalerCorebankApi,
  TalerErrorCode,
} from "@gnu-taler/taler-util";
import {
  ButtonBetter,
  LocalNotificationBanner,
  notifyInfo,
  RouteDefinition,
  useBankCoreApiContext,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h, VNode } from "preact";
import { useState } from "preact/hooks";
import { useSessionState } from "../hooks/session.js";
import { ConversionRateClassForm } from "./admin/ConversionRateClassForm.js";

const TALER_SCREEN_ID = 13;
interface Props {
  routeCancel: RouteDefinition;
  onCreated: (id: number) => void;
}
export function NewConversionRateClass({
  routeCancel,
  onCreated,
}: Props): VNode {
  const { i18n } = useTranslationContext();
  const { state: credentials } = useSessionState();
  const token =
    credentials.status !== "loggedIn" ? undefined : credentials.token;
  const {
    lib: { bank: api },
  } = useBankCoreApiContext();

  const [notification, safeFunctionHandler] = useLocalNotificationBetter();

  const [submitData, setSubmitData] = useState<
    TalerCorebankApi.ConversionRateClassInput | undefined
  >();

  const create = safeFunctionHandler(
    (token: AccessToken, data: TalerCorebankApi.ConversionRateClassInput) =>
      api.createConversionRateClass(token, data),
    !submitData || !token ? undefined : [token, submitData],
  );
  create.onSuccess = (success) => {
    notifyInfo(i18n.str`Conversion rate class created.`);
    onCreated(success.conversion_rate_class_id);
  };
  create.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`The rights to change the account are not sufficient`;
      case HttpStatusCode.Forbidden:
        return i18n.str`Wrong credentials`;
      case HttpStatusCode.NotFound:
        return i18n.str`Account not found`;
      case HttpStatusCode.NotImplemented:
        return i18n.str`Not implemented`;
      case TalerErrorCode.BANK_NAME_REUSE:
        return i18n.str`The name of the conversion is already used.`;
    }
  };

  return (
    <div class="grid grid-cols-1 gap-x-8 gap-y-8 pt-6 md:grid-cols-3 bg-gray-100 my-4 px-4 pb-4 rounded-lg">
      <LocalNotificationBanner notification={notification} />

      <div class="px-4 sm:px-0">
        <h2 class="text-base font-semibold leading-7 text-gray-900">
          <i18n.Translate>New conversion rate class</i18n.Translate>
        </h2>
      </div>

      <ConversionRateClassForm onChange={setSubmitData}>
        <div class="flex items-center justify-between gap-x-6 border-t border-gray-900/10 px-4 py-4 sm:px-8">
          <a
            href={routeCancel.url({})}
            name="cancel"
            class="text-sm font-semibold leading-6 text-gray-900"
          >
            <i18n.Translate>Cancel</i18n.Translate>
          </a>
          <ButtonBetter
            type="submit"
            name="create"
            class="disabled:opacity-50 disabled:cursor-default cursor-pointer rounded-md bg-indigo-600 px-3 py-2 text-sm font-semibold text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
            onClick={create}
          >
            <i18n.Translate>Create</i18n.Translate>
          </ButtonBetter>
        </div>
      </ConversionRateClassForm>
    </div>
  );
}
