/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  AbsoluteTime,
  Amounts,
  NotificationType,
  Transaction,
  TransactionMajorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { Fragment, h, JSX, VNode } from "preact";
import { useEffect } from "preact/hooks";
import { useBackendContext } from "../context/backend.js";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { useAsyncAsHook } from "../hooks/useAsyncAsHook.js";
import { Avatar } from "../mui/Avatar.js";
import { Grid } from "../mui/Grid.js";
import { Typography } from "../mui/Typography.js";
import Banner from "./Banner.js";
import { Time } from "./Time.js";

interface Props extends JSX.HTMLAttributes {
  goToTransaction?: (id: string) => Promise<void>;
  goToURL: (url: string) => void;
}

/**
 * this cache will save the tx from the previous render
 */
const cache = { tx: [] as Transaction[] };

export function PendingTransactions({
  goToTransaction,
  goToURL,
}: Props): VNode {
  const api = useBackendContext();
  const state = useAsyncAsHook(() =>
    api.wallet.call(WalletApiOperation.GetTransactionsV2, {}),
  );

  useEffect(() => {
    return api.listener.onUpdateNotification(
      [NotificationType.TransactionStateTransition],
      state?.retry,
    );
  });

  const transactions =
    !state || state.hasError
      ? cache.tx
      : state.response.transactions.filter(
          (t) => t.txState.major === TransactionMajorState.Pending,
        );

  if (state && !state.hasError) {
    cache.tx = transactions;
  }
  if (!transactions.length) {
    return <Fragment />;
  }
  return (
    <PendingTransactionsView
      goToTransaction={goToTransaction}
      goToURL={goToURL}
      transactions={transactions}
    />
  );
}

export function PendingTransactionsView({
  transactions,
  goToTransaction,
  goToURL,
}: {
  goToTransaction?: (id: string) => Promise<void>;
  goToURL: (id: string) => void;
  transactions: Transaction[];
}): VNode {
  const { i18n } = useTranslationContext();
  const kycTransaction = transactions.find((tx) => tx.kycUrl);
  if (kycTransaction) {
    return (
      <div
        style={{
          backgroundColor: "#fff3cd",
          color: "#664d03",
          display: "flex",
          justifyContent: "center",
        }}
      >
        <Banner
          titleHead={i18n.str`KYC requirement`}
          style={{
            backgroundColor: "lightred",
            maxHeight: 150,
            padding: 8,
            flexGrow: 1, //#fff3cd //#ffecb5
            maxWidth: 500,
            overflowY: transactions.length > 3 ? "scroll" : "hidden",
          }}
        >
          <Grid
            container
            item
            xs={1}
            wrap="nowrap"
            role="button"
            spacing={1}
            alignItems="center"
            onClick={() => {
              goToURL(kycTransaction.kycUrl ?? "#");
            }}
          >
            <Grid item>
              <Typography inline bold>
                One or more transaction require a KYC step to complete
              </Typography>
            </Grid>
          </Grid>
        </Banner>
      </div>
    );
  }

  if (!goToTransaction) return <Fragment />;

  return (
    <div
      style={{
        backgroundColor: "lightcyan",
        display: "flex",
        justifyContent: "center",
      }}
    >
      <Banner
        titleHead={i18n.str`PENDING OPERATIONS`}
        style={{
          backgroundColor: "lightcyan",
          maxHeight: 150,
          padding: 8,
          flexGrow: 1,
          maxWidth: 500,
          overflowY: transactions.length > 3 ? "scroll" : "hidden",
        }}
      >
        {transactions.map((t, i) => {
          const amount = Amounts.parseOrThrow(t.amountEffective);
          return (
            <Grid
              container
              item
              xs={1}
              key={i}
              wrap="nowrap"
              role="button"
              spacing={1}
              alignItems="center"
              onClick={() => {
                goToTransaction(t.transactionId);
              }}
            >
              <Grid item xs={"auto"}>
                <Avatar
                  style={{
                    border: "solid blue 1px",
                    color: "blue",
                    boxSizing: "border-box",
                  }}
                >
                  {t.type.substring(0, 1)}
                </Avatar>
              </Grid>

              <Grid item>
                <Typography inline bold>
                  {amount.currency} {Amounts.stringifyValue(amount)}
                </Typography>
                &nbsp;-&nbsp;
                <Time
                  timestamp={AbsoluteTime.fromPreciseTimestamp(t.timestamp)}
                  format="dd MMMM yyyy"
                />
              </Grid>
            </Grid>
          );
        })}
      </Banner>
    </div>
  );
}

export default PendingTransactions;
